#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <assert.h>
#include <argp.h>
#include <string.h>
#include <poll.h>
#include <sys/time.h>
#include <unistd.h>

#include "control/device_sync.h"
#include "control/sync_control.h"
#include "util/device_sync_parser.h"
#include "util/logger.h"

/*
 * implements the main loop
 */
static error_code_t device_sync_mainloop(void);

//----------------------------------------------- device sync api members ----------------------------------------------
error_code_t device_sync_init(int argc,char *argv[])
{
	error_code_t result;
	logger_loglevel_t loglevel;

	loglevel=device_sync_parser_parse_loglevel(argc,argv);
	logger_init(loglevel);

	sync_control_init();

	result=device_sync_parser_parse_args(argc,argv);

	if (result==RESULT_OK)
		result=device_sync_parser_parse_config();

	logger_log_debug("DEVICE_SYNC - init done (result=%d).",result);
	return result;
}

error_code_t device_sync_start(void)
{
	return device_sync_mainloop();
}

void device_sync_deinit(bool print_missing)
{  
	sync_control_deinit(print_missing);
	logger_log_debug("DEVICE_SYNC -> deinit complete.");
}
//-------------------------------------------------------------------------------------------------------------------------

//------------------------------------------------device sync private members ------------------------------------------
static error_code_t device_sync_mainloop(void)
{
	struct pollfd pollfds;
	int poll_timeout;
	int poll_result;
	int state;
	error_code_t error_status=RESULT_OK;

	pollfds.events=POLLIN;
	pollfds.fd=sync_control_get_pollfd();

	logger_log_debug("DEVICE_SYNC -> Entering the mainloop.");

	//activating timeout
	sync_control_activate_timeout((int)device_sync_parser_get_timeout_ms());

	//a mask telling us if we have still devices and/or attributes in the lists
	state=sync_control_get_state();

	while(state!=SYNC_CO_STATE_DONE && error_status==RESULT_OK)
	{
		poll_timeout=sync_control_get_remaining_timeout_time();
		//in case we are waiting for attributes only, we might need a smaller timeout
		if ((state & SYNC_CO_STATE_WAITING_FOR_DEVICES)==0 && (poll_timeout > ATTRIBUTE_POLL_TIME_MS || poll_timeout==-1))
		{
			poll_timeout=ATTRIBUTE_POLL_TIME_MS;
		}

		logger_log_debug("DEVICE_SYNC -> Waiting for devices now, timeout: %d ms.", poll_timeout);

		//lets poll if we have something in the device list, otherwise, just sleep for a while
		if ((state & SYNC_CO_STATE_WAITING_FOR_DEVICES)!=0)
		{
			poll_result=poll(&pollfds,1,poll_timeout);
		}
		else
		{
			usleep((unsigned int)(poll_timeout*1000));
			//we are timining out
			poll_result=0;
		}

		logger_log_debug("DEVICE_SYNC -> Mainloop got woken up.");

		if (poll_result<0)
			continue;
		else if (poll_result==0)
		{
			if (sync_control_get_remaining_timeout_time()>0)
				error_status=sync_control_check_attributes();
			else
				error_status=RESULT_TIMEOUT_EXPIRED;
		}
		else
			error_status=sync_control_on_event();

		//a mask telling us if we have still devices and/or attributes in the lists
		state=sync_control_get_state();
	}

	logger_log_debug("DEVICE_SYNC -> Leaving the main loop. Result: %d",error_status);

	return error_status;
}
//-------------------------------------------------------------------------------------------------------------------------
